/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.numbers.fraction;

import java.io.Serializable;
import org.apache.commons.numbers.core.ArithmeticUtils;
import org.apache.commons.numbers.core.NativeOperators;
import org.apache.commons.numbers.fraction.FractionException;

public final class Fraction
extends Number
implements Comparable<Fraction>,
NativeOperators<Fraction>,
Serializable {
    public static final Fraction ZERO = new Fraction(0);
    public static final Fraction ONE = new Fraction(1);
    private static final long serialVersionUID = 20190701L;
    private static final double DEFAULT_EPSILON = 1.0E-5;
    private static final int DEFAULT_MAX_ITERATIONS = 100;
    private static final String NOT_FINITE = "Not finite: ";
    private static final long OVERFLOW = 0x80000000L;
    private final int numerator;
    private final int denominator;

    private Fraction(int num, int den) {
        if (den == 0) {
            throw new FractionException("Denominator must be different from 0");
        }
        if (num == den) {
            this.numerator = 1;
            this.denominator = 1;
        } else {
            int q;
            int p;
            if (((num | den) & 1) == 0) {
                p = num >> 1;
                q = den >> 1;
            } else {
                p = num;
                q = den;
            }
            int d = ArithmeticUtils.gcd((int)p, (int)q);
            this.numerator = p / d;
            this.denominator = q / d;
        }
    }

    private Fraction(int num) {
        this.numerator = num;
        this.denominator = 1;
    }

    private Fraction(double value, double epsilon, int maxDenominator, int maxIterations) {
        int den;
        int num;
        if (!Double.isFinite(value)) {
            throw new IllegalArgumentException(NOT_FINITE + value);
        }
        double absValue = Math.abs(value);
        double r0 = absValue;
        long a0 = (long)Math.floor(r0);
        if (a0 > 0x80000000L) {
            throw new FractionException("Overflow trying to convert %s to fraction (%d/%d)", value, a0, 1);
        }
        if (r0 - (double)a0 <= epsilon) {
            int num2 = (int)a0;
            int den2 = 1;
            if ((double)Math.signum(num2) != Math.signum(value)) {
                if (num2 == Integer.MIN_VALUE) {
                    den2 = -den2;
                } else {
                    num2 = -num2;
                }
            }
            this.numerator = num2;
            this.denominator = den2;
            return;
        }
        long maxDen = Math.abs((long)maxDenominator);
        long p0 = 1L;
        long q0 = 0L;
        long p1 = a0;
        long q1 = 1L;
        long p2 = 0L;
        long q2 = 1L;
        int n = 0;
        boolean stop = false;
        do {
            ++n;
            double r1 = 1.0 / (r0 - (double)a0);
            long a1 = (long)Math.floor(r1);
            p2 = a1 * p1 + p0;
            q2 = a1 * q1 + q0;
            if (Long.compareUnsigned(p2, 0x80000000L) > 0 || Long.compareUnsigned(q2, 0x80000000L) > 0) {
                if (epsilon == 0.0) {
                    p2 = p1;
                    q2 = q1;
                    break;
                }
                throw new FractionException("Overflow trying to convert %s to fraction (%d/%d)", value, p2, q2);
            }
            double convergent = (double)p2 / (double)q2;
            if (n < maxIterations && Math.abs(convergent - absValue) > epsilon && q2 < maxDen) {
                p0 = p1;
                p1 = p2;
                q0 = q1;
                q1 = q2;
                a0 = a1;
                r0 = r1;
                continue;
            }
            stop = true;
        } while (!stop);
        if (n >= maxIterations) {
            throw new FractionException("Unable to convert %s to fraction after %d iterations", value, maxIterations);
        }
        if (q2 <= maxDen) {
            num = (int)p2;
            den = (int)q2;
        } else {
            num = (int)p1;
            den = (int)q1;
        }
        if ((double)(Math.signum(num) * Math.signum(den)) != Math.signum(value)) {
            if (num == Integer.MIN_VALUE) {
                den = -den;
            } else {
                num = -num;
            }
        }
        this.numerator = num;
        this.denominator = den;
    }

    public static Fraction from(double value) {
        return Fraction.from(value, 1.0E-5, 100);
    }

    public static Fraction from(double value, double epsilon, int maxIterations) {
        if (value == 0.0) {
            return ZERO;
        }
        if (maxIterations < 1) {
            throw new IllegalArgumentException("Max iterations must be strictly positive: " + maxIterations);
        }
        if (epsilon >= 0.0) {
            return new Fraction(value, epsilon, Integer.MIN_VALUE, maxIterations);
        }
        throw new IllegalArgumentException("Epsilon must be positive: " + maxIterations);
    }

    public static Fraction from(double value, int maxDenominator) {
        if (value == 0.0) {
            return ZERO;
        }
        if (maxDenominator == 0) {
            throw new IllegalArgumentException("Denominator must be different from 0");
        }
        return new Fraction(value, 0.0, maxDenominator, 100);
    }

    public static Fraction of(int num) {
        if (num == 0) {
            return ZERO;
        }
        return new Fraction(num);
    }

    public static Fraction of(int num, int den) {
        if (num == 0) {
            return ZERO;
        }
        return new Fraction(num, den);
    }

    public static Fraction parse(String s) {
        String stripped = s.replace(",", "");
        int slashLoc = stripped.indexOf(47);
        if (slashLoc == -1) {
            return Fraction.of(Integer.parseInt(stripped.trim()));
        }
        int num = Integer.parseInt(stripped.substring(0, slashLoc).trim());
        int denom = Integer.parseInt(stripped.substring(slashLoc + 1).trim());
        return Fraction.of(num, denom);
    }

    public Fraction zero() {
        return ZERO;
    }

    public Fraction one() {
        return ONE;
    }

    public int getNumerator() {
        return this.numerator;
    }

    public int getDenominator() {
        return this.denominator;
    }

    public int signum() {
        return Integer.signum(this.numerator) * Integer.signum(this.denominator);
    }

    public Fraction abs() {
        return this.signum() >= 0 ? this : this.negate();
    }

    public Fraction negate() {
        return this.numerator == Integer.MIN_VALUE ? new Fraction(this.numerator, -this.denominator) : new Fraction(-this.numerator, this.denominator);
    }

    public Fraction reciprocal() {
        return new Fraction(this.denominator, this.numerator);
    }

    @Override
    public double doubleValue() {
        return (double)this.numerator / (double)this.denominator;
    }

    @Override
    public float floatValue() {
        return (float)this.doubleValue();
    }

    @Override
    public int intValue() {
        return (int)this.doubleValue();
    }

    @Override
    public long longValue() {
        return (long)this.numerator / (long)this.denominator;
    }

    public Fraction add(int value) {
        if (value == 0) {
            return this;
        }
        if (this.isZero()) {
            return new Fraction(value);
        }
        long num = (long)value * (long)this.denominator;
        return Fraction.of(Math.toIntExact((long)this.numerator + num), this.denominator);
    }

    public Fraction add(Fraction value) {
        return this.addSub(value, true);
    }

    public Fraction subtract(int value) {
        if (value == 0) {
            return this;
        }
        if (this.isZero()) {
            return value == Integer.MIN_VALUE ? new Fraction(Integer.MIN_VALUE, -1) : new Fraction(-value);
        }
        long num = (long)value * (long)this.denominator;
        return Fraction.of(Math.toIntExact((long)this.numerator - num), this.denominator);
    }

    public Fraction subtract(Fraction value) {
        return this.addSub(value, false);
    }

    private Fraction addSub(Fraction value, boolean isAdd) {
        if (value.isZero()) {
            return this;
        }
        if (this.isZero()) {
            return isAdd ? value : value.negate();
        }
        int d1 = ArithmeticUtils.gcd((int)this.denominator, (int)value.denominator);
        long uvp = (long)this.numerator * (long)(value.denominator / d1);
        long upv = (long)value.numerator * (long)(this.denominator / d1);
        long t = isAdd ? uvp + upv : uvp - upv;
        long d2 = ArithmeticUtils.gcd((long)t, (long)d1);
        return Fraction.of(Math.toIntExact(t / d2), Math.multiplyExact(this.denominator / d1, value.denominator / (int)d2));
    }

    public Fraction multiply(int value) {
        if (value == 0 || this.isZero()) {
            return ZERO;
        }
        int d2 = ArithmeticUtils.gcd((int)value, (int)this.denominator);
        return new Fraction(Math.multiplyExact(this.numerator, value / d2), this.denominator / d2);
    }

    public Fraction multiply(Fraction value) {
        if (value.isZero() || this.isZero()) {
            return ZERO;
        }
        return this.multiply(value.numerator, value.denominator);
    }

    private Fraction multiply(int num, int den) {
        int d1 = ArithmeticUtils.gcd((int)this.numerator, (int)den);
        int d2 = ArithmeticUtils.gcd((int)num, (int)this.denominator);
        return new Fraction(Math.multiplyExact(this.numerator / d1, num / d2), Math.multiplyExact(this.denominator / d2, den / d1));
    }

    public Fraction divide(int value) {
        if (value == 0) {
            throw new FractionException("The value to divide by must not be zero");
        }
        if (this.isZero()) {
            return ZERO;
        }
        int d1 = ArithmeticUtils.gcd((int)this.numerator, (int)value);
        return new Fraction(this.numerator / d1, Math.multiplyExact(this.denominator, value / d1));
    }

    public Fraction divide(Fraction value) {
        if (value.isZero()) {
            throw new FractionException("The value to divide by must not be zero");
        }
        if (this.isZero()) {
            return ZERO;
        }
        return this.multiply(value.denominator, value.numerator);
    }

    public Fraction pow(int exponent) {
        if (exponent == 1) {
            return this;
        }
        if (exponent == 0) {
            return ONE;
        }
        if (this.isZero()) {
            if (exponent < 0) {
                throw new FractionException("Denominator must be different from 0");
            }
            return ZERO;
        }
        if (exponent > 0) {
            return new Fraction(ArithmeticUtils.pow((int)this.numerator, (int)exponent), ArithmeticUtils.pow((int)this.denominator, (int)exponent));
        }
        if (exponent == -1) {
            return this.reciprocal();
        }
        if (exponent == Integer.MIN_VALUE) {
            return new Fraction(ArithmeticUtils.pow((int)this.denominator, (int)Integer.MAX_VALUE) * this.denominator, ArithmeticUtils.pow((int)this.numerator, (int)Integer.MAX_VALUE) * this.numerator);
        }
        return new Fraction(ArithmeticUtils.pow((int)this.denominator, (int)(-exponent)), ArithmeticUtils.pow((int)this.numerator, (int)(-exponent)));
    }

    public String toString() {
        String str = this.isZero() ? "0" : (this.denominator == 1 ? Integer.toString(this.numerator) : this.numerator + " / " + this.denominator);
        return str;
    }

    @Override
    public int compareTo(Fraction other) {
        int rds;
        int rns;
        int rhsSigNum;
        int lds;
        int lns = Integer.signum(this.numerator);
        int lhsSigNum = lns * (lds = Integer.signum(this.denominator));
        if (lhsSigNum != (rhsSigNum = (rns = Integer.signum(other.numerator)) * (rds = Integer.signum(other.denominator)))) {
            return lhsSigNum > rhsSigNum ? 1 : -1;
        }
        if (lhsSigNum == 0) {
            return 0;
        }
        long nOd = (long)this.numerator * (long)lns * (long)other.denominator * (long)rds;
        long dOn = (long)this.denominator * (long)lds * (long)other.numerator * (long)rns;
        return Long.compare(nOd, dOn);
    }

    public boolean equals(Object other) {
        if (this == other) {
            return true;
        }
        if (other instanceof Fraction) {
            Fraction rhs = (Fraction)other;
            if (this.signum() == rhs.signum()) {
                return Math.abs(this.numerator) == Math.abs(rhs.numerator) && Math.abs(this.denominator) == Math.abs(rhs.denominator);
            }
        }
        return false;
    }

    public int hashCode() {
        int numS = Integer.signum(this.numerator);
        int denS = Integer.signum(this.denominator);
        return (31 * (31 + this.numerator * numS) + this.denominator * denS) * numS * denS;
    }

    private boolean isZero() {
        return this.numerator == 0;
    }
}

